import express from "express";
import cors from "cors";
import morgan from "morgan";
import mongoose from "mongoose";
import bcrypt from "bcryptjs";
import jwt from "jsonwebtoken";

const app = express();

// Middleware
app.use(cors());
app.use(morgan("combined"));
app.use(express.json());

// MongoDB Connection
mongoose.connect(
  process.env.MONGODB_URI || "mongodb://localhost:27017/portfolio-cms",
  {
    useNewUrlParser: true,
    useUnifiedTopology: true
  }
);

const db = mongoose.connection;
db.on("error", console.error.bind(console, "MongoDB connection error:"));
db.once("open", () => {
  console.log("Connected to MongoDB");
});

// Mongoose Schemas and Models

// User Schema
const userSchema = new mongoose.Schema({
  email: { type: String, required: true, unique: true },
  password: { type: String, required: true },
  name: { type: String, required: true },
  role: { type: String, default: "admin" },
  createdAt: { type: Date, default: Date.now }
});

const User = mongoose.model("User", userSchema);

// Settings Schema (Cloudinary config)
const settingsSchema = new mongoose.Schema({
  key: { type: String, default: "admin_settings", unique: true },
  cloudinaryCloudName: String,
  cloudinaryUploadPreset: String,
  cloudinaryApiKey: String,
  updatedAt: { type: Date, default: Date.now }
});

const Settings = mongoose.model("Settings", settingsSchema);

// Hero Image Schema
const heroImageSchema = new mongoose.Schema({
  url: { type: String, required: true },
  title: String,
  subtitle: String,
  order: { type: Number, default: 0 },
  createdAt: { type: Date, default: Date.now }
});

const HeroImage = mongoose.model("HeroImage", heroImageSchema);

// Gallery Image Schema
const galleryImageSchema = new mongoose.Schema({
  url: { type: String, required: true },
  title: String,
  description: String,
  category: String,
  order: { type: Number, default: 0 },
  createdAt: { type: Date, default: Date.now }
});

const GalleryImage = mongoose.model("GalleryImage", galleryImageSchema);

// Contact Info Schema
const contactInfoSchema = new mongoose.Schema({
  key: { type: String, default: "contact_info", unique: true },
  email: String,
  phone: String,
  address: String,
  socialMedia: {
    facebook: String,
    instagram: String,
    twitter: String,
    linkedin: String
  },
  updatedAt: { type: Date, default: Date.now }
});

const ContactInfo = mongoose.model("ContactInfo", contactInfoSchema);

// Portfolio Item Schema
const portfolioItemSchema = new mongoose.Schema({
  title: { type: String, required: true },
  description: String,
  images: [String],
  category: String,
  tags: [String],
  url: String,
  order: { type: Number, default: 0 },
  createdAt: { type: Date, default: Date.now }
});

const PortfolioItem = mongoose.model("PortfolioItem", portfolioItemSchema);

// Contact Message Schema
const contactMessageSchema = new mongoose.Schema({
  name: { type: String, required: true },
  email: { type: String, required: true },
  phone: String,
  message: { type: String, required: true },
  read: { type: Boolean, default: false },
  createdAt: { type: Date, default: Date.now }
});

const ContactMessage = mongoose.model("ContactMessage", contactMessageSchema);

// Auth middleware
const requireAuth = async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;

    if (!authHeader) {
      return res.status(401).json({ error: "Unauthorized" });
    }

    const token = authHeader.split(" ")[1];
    const decoded = jwt.verify(
      token,
      process.env.JWT_SECRET || "your-secret-key"
    );

    const user = await User.findById(decoded.userId);

    if (!user) {
      return res.status(401).json({ error: "Unauthorized" });
    }

    req.user = user;
    next();
  } catch (error) {
    return res.status(401).json({ error: "Unauthorized" });
  }
};

// Create admin user (first time setup)
app.post("/api/admin/signup", async (req, res) => {
  try {
    const { email, password, name } = req.body;

    // Check if user already exists
    const existingUser = await User.findOne({ email });
    if (existingUser) {
      return res.status(400).json({ error: "User already exists" });
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Create user
    const user = new User({
      email,
      password: hashedPassword,
      name,
      role: "admin"
    });

    await user.save();

    // Generate JWT token
    const token = jwt.sign(
      { userId: user._id, email: user.email },
      process.env.JWT_SECRET || "your-secret-key",
      { expiresIn: "7d" }
    );

    return res.json({
      success: true,
      user: {
        id: user._id,
        email: user.email,
        name: user.name
      },
      token
    });
  } catch (error) {
    console.log(`Error in admin signup: ${error}`);
    return res.status(500).json({ error: "Failed to create admin user" });
  }
});

// Admin login
app.post("/api/admin/login", async (req, res) => {
  try {
    const { email, password } = req.body;
    // Debugging logs: print request body so client payload can be verified
    console.log("[/api/admin/login] request body:", req.body);

    const user = await User.findOne({ email });
    console.log("[/api/admin/login] found user:", user ? user.email : null);
    if (!user) {
      return res.status(401).json({ error: "Invalid credentials" });
    }

    let isValidPassword = false;
    try {
      isValidPassword = await bcrypt.compare(password, user.password);
    } catch (pwErr) {
      console.error("[/api/admin/login] bcrypt.compare error:", pwErr);
      // treat as invalid credentials but log full error
    }
    if (!isValidPassword) {
      return res.status(401).json({ error: "Invalid credentials" });
    }

    const token = jwt.sign(
      { userId: user._id, email: user.email },
      process.env.JWT_SECRET || "your-secret-key",
      { expiresIn: "7d" }
    );

    return res.json({
      success: true,
      user: {
        id: user._id,
        email: user.email,
        name: user.name
      },
      token
    });
  } catch (error) {
    console.error(
      `Error in admin login:`,
      error && error.stack ? error.stack : error
    );
    return res.status(500).json({ error: "Login failed" });
  }
});

// Admin login verification
app.post("/api/admin/verify", requireAuth, async (req, res) => {
  return res.json({
    success: true,
    user: {
      id: req.user._id,
      email: req.user.email,
      name: req.user.name
    }
  });
});

// Get settings (Cloudinary config)
app.get("/api/settings", requireAuth, async (req, res) => {
  try {
    let settings = await Settings.findOne({ key: "admin_settings" });
    if (!settings) {
      settings = new Settings({ key: "admin_settings" });
      await settings.save();
    }
    return res.json({ settings });
  } catch (error) {
    console.log(`Error fetching settings: ${error}`);
    return res.status(500).json({ error: "Failed to fetch settings" });
  }
});

// Update settings
app.post("/api/settings", requireAuth, async (req, res) => {
  try {
    const settingsData = req.body;
    let settings = await Settings.findOne({ key: "admin_settings" });

    if (!settings) {
      settings = new Settings({ key: "admin_settings", ...settingsData });
    } else {
      Object.assign(settings, settingsData);
      settings.updatedAt = new Date();
    }

    await settings.save();
    return res.json({ success: true });
  } catch (error) {
    console.log(`Error updating settings: ${error}`);
    return res.status(500).json({ error: "Failed to update settings" });
  }
});

// Get Cloudinary config (for frontend direct upload)
app.get("/api/cloudinary-config", requireAuth, async (req, res) => {
  try {
    const settings = await Settings.findOne({ key: "admin_settings" });

    if (!settings?.cloudinaryCloudName || !settings?.cloudinaryUploadPreset) {
      return res.status(400).json({
        error: "Cloudinary not configured. Please set up in Settings."
      });
    }

    return res.json({
      cloudName: settings.cloudinaryCloudName,
      uploadPreset: settings.cloudinaryUploadPreset
    });
  } catch (error) {
    console.log(`Error fetching Cloudinary config: ${error}`);
    return res.status(500).json({ error: "Failed to fetch Cloudinary config" });
  }
});

// Get hero images
app.get("/api/hero-images", async (req, res) => {
  try {
    const images = await HeroImage.find().sort({ order: 1 });
    return res.json({ images });
  } catch (error) {
    console.log(`Error fetching hero images: ${error}`);
    return res.status(500).json({ error: "Failed to fetch hero images" });
  }
});

// Add single hero image (called after frontend uploads to Cloudinary)
app.post("/api/hero-images/add", requireAuth, async (req, res) => {
  try {
    const { url, title, subtitle, order } = req.body;

    if (!url) {
      return res.status(400).json({ error: "Image URL is required" });
    }

    const heroImage = new HeroImage({
      url,
      title: title || "",
      subtitle: subtitle || "",
      order: order || 0
    });

    await heroImage.save();
    return res.json({ success: true, image: heroImage });
  } catch (error) {
    console.log(`Error adding hero image: ${error}`);
    return res.status(500).json({ error: "Failed to add hero image" });
  }
});

// Update hero images (bulk update for reordering)
app.post("/api/hero-images", requireAuth, async (req, res) => {
  try {
    const { images } = req.body;

    // Delete all existing images
    await HeroImage.deleteMany({});

    // Insert new images
    if (images && images.length > 0) {
      await HeroImage.insertMany(images);
    }

    return res.json({ success: true });
  } catch (error) {
    console.log(`Error updating hero images: ${error}`);
    return res.status(500).json({ error: "Failed to update hero images" });
  }
});

// Delete hero image
app.delete("/api/hero-images/:id", requireAuth, async (req, res) => {
  try {
    const imageId = req.params.id;
    await HeroImage.findByIdAndDelete(imageId);
    return res.json({ success: true });
  } catch (error) {
    console.log(`Error deleting hero image: ${error}`);
    return res.status(500).json({ error: "Failed to delete hero image" });
  }
});

// Get gallery images
app.get("/api/gallery", async (req, res) => {
  try {
    const images = await GalleryImage.find().sort({ order: 1 });
    return res.json({ images });
  } catch (error) {
    console.log(`Error fetching gallery images: ${error}`);
    return res.status(500).json({ error: "Failed to fetch gallery images" });
  }
});

// Add single gallery image (called after frontend uploads to Cloudinary)
app.post("/api/gallery/add", requireAuth, async (req, res) => {
  try {
    const { url, title, description, category, order } = req.body;

    if (!url) {
      return res.status(400).json({ error: "Image URL is required" });
    }

    const galleryImage = new GalleryImage({
      url,
      title: title || "",
      description: description || "",
      category: category || "",
      order: order || 0
    });

    await galleryImage.save();
    return res.json({ success: true, image: galleryImage });
  } catch (error) {
    console.log(`Error adding gallery image: ${error}`);
    return res.status(500).json({ error: "Failed to add gallery image" });
  }
});

// Update gallery images (bulk update for reordering)
app.post("/api/gallery", requireAuth, async (req, res) => {
  try {
    const { images } = req.body;

    // Delete all existing images
    await GalleryImage.deleteMany({});

    // Insert new images
    if (images && images.length > 0) {
      await GalleryImage.insertMany(images);
    }

    return res.json({ success: true });
  } catch (error) {
    console.log(`Error updating gallery images: ${error}`);
    return res.status(500).json({ error: "Failed to update gallery images" });
  }
});

// Delete gallery image
app.delete("/api/gallery/:id", requireAuth, async (req, res) => {
  try {
    const imageId = req.params.id;
    await GalleryImage.findByIdAndDelete(imageId);
    return res.json({ success: true });
  } catch (error) {
    console.log(`Error deleting gallery image: ${error}`);
    return res.status(500).json({ error: "Failed to delete gallery image" });
  }
});

// Get contact info
app.get("/api/contact", async (req, res) => {
  try {
    let contact = await ContactInfo.findOne({ key: "contact_info" });
    if (!contact) {
      contact = new ContactInfo({ key: "contact_info" });
      await contact.save();
    }
    return res.json({ contact });
  } catch (error) {
    console.log(`Error fetching contact info: ${error}`);
    return res.status(500).json({ error: "Failed to fetch contact info" });
  }
});

// Update contact info
app.post("/api/contact", requireAuth, async (req, res) => {
  try {
    const contactData = req.body;
    let contact = await ContactInfo.findOne({ key: "contact_info" });

    if (!contact) {
      contact = new ContactInfo({ key: "contact_info", ...contactData });
    } else {
      Object.assign(contact, contactData);
      contact.updatedAt = new Date();
    }

    await contact.save();
    return res.json({ success: true });
  } catch (error) {
    console.log(`Error updating contact info: ${error}`);
    return res.status(500).json({ error: "Failed to update contact info" });
  }
});

// Get portfolio items
app.get("/api/portfolio", async (req, res) => {
  try {
    const portfolio = await PortfolioItem.find().sort({ order: 1 });
    return res.json({ portfolio });
  } catch (error) {
    console.log(`Error fetching portfolio items: ${error}`);
    return res.status(500).json({ error: "Failed to fetch portfolio items" });
  }
});

// Add single portfolio item (called after frontend uploads to Cloudinary)
app.post("/api/portfolio/add", requireAuth, async (req, res) => {
  try {
    const { title, description, images, category, tags, url, order } = req.body;

    if (!title) {
      return res.status(400).json({ error: "Title is required" });
    }

    const portfolioItem = new PortfolioItem({
      title,
      description: description || "",
      images: images || [],
      category: category || "",
      tags: tags || [],
      url: url || "",
      order: order || 0
    });

    await portfolioItem.save();
    return res.json({ success: true, item: portfolioItem });
  } catch (error) {
    console.log(`Error adding portfolio item: ${error}`);
    return res.status(500).json({ error: "Failed to add portfolio item" });
  }
});

// Update portfolio items (bulk update for reordering)
app.post("/api/portfolio", requireAuth, async (req, res) => {
  try {
    const { portfolio } = req.body;

    // Delete all existing portfolio items
    await PortfolioItem.deleteMany({});

    // Insert new portfolio items
    if (portfolio && portfolio.length > 0) {
      await PortfolioItem.insertMany(portfolio);
    }

    return res.json({ success: true });
  } catch (error) {
    console.log(`Error updating portfolio items: ${error}`);
    return res.status(500).json({ error: "Failed to update portfolio items" });
  }
});

// Delete portfolio item
app.delete("/api/portfolio/:id", requireAuth, async (req, res) => {
  try {
    const itemId = req.params.id;
    await PortfolioItem.findByIdAndDelete(itemId);
    return res.json({ success: true });
  } catch (error) {
    console.log(`Error deleting portfolio item: ${error}`);
    return res.status(500).json({ error: "Failed to delete portfolio item" });
  }
});

// Submit contact message
app.post("/api/messages", async (req, res) => {
  try {
    const { name, email, phone, message } = req.body;

    const newMessage = new ContactMessage({
      name,
      email,
      phone,
      message,
      read: false
    });

    await newMessage.save();

    return res.json({ success: true });
  } catch (error) {
    console.log(`Error submitting contact message: ${error}`);
    return res.status(500).json({ error: "Failed to submit message" });
  }
});

// Get contact messages (admin only)
app.get("/api/messages", requireAuth, async (req, res) => {
  try {
    const messages = await ContactMessage.find().sort({ createdAt: -1 });
    return res.json({ messages });
  } catch (error) {
    console.log(`Error fetching messages: ${error}`);
    return res.status(500).json({ error: "Failed to fetch messages" });
  }
});

// Mark message as read
app.post("/api/messages/:id/read", requireAuth, async (req, res) => {
  try {
    const messageId = req.params.id;
    const message = await ContactMessage.findById(messageId);

    if (!message) {
      return res.status(404).json({ error: "Message not found" });
    }

    message.read = true;
    await message.save();

    return res.json({ success: true });
  } catch (error) {
    console.log(`Error marking message as read: ${error}`);
    return res.status(500).json({ error: "Failed to update message" });
  }
});

// Delete message
app.delete("/api/messages/:id", requireAuth, async (req, res) => {
  try {
    const messageId = req.params.id;
    await ContactMessage.findByIdAndDelete(messageId);

    return res.json({ success: true });
  } catch (error) {
    console.log(`Error deleting message: ${error}`);
    return res.status(500).json({ error: "Failed to delete message" });
  }
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});
